/************************************************************************
 * NAME:	svd20.c
 *
 * DESCR:	Support for version 2.0 of the SVD format.  You should
 *		also look at svd.[ch] and svd[0-9][0-9].[ch] which have
 *		code for the different versions of the SVD format.
 *		These were split out from this file to eliminate
 *		confusion and inadvertant changes to the old formats.
 ************************************************************************/
#include <sys/types.h>
#include <stdio.h>

#include "standard.h"
#include "floppy.h"
#include "error.h"
#include "svd.h"
#include "svd20.h"
#include "apple.h"

#define THIS_TRACK	floppy->side[side][track]
#define THIS_SECTOR	THIS_TRACK.sector[sector]


/************************************************************************
 * NAME:	svd_read_20header()
 *
 * DESCR:	Reads the v2.0 header information.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
svd_read_20header(int fd, int *sectors, int *tracks, int *sides, int *secsize, int *wprot)
{
    char    linebuffer[80];
    char   *ptr;
    int	    count;

    count = svd_read_line(fd,linebuffer,sizeof(linebuffer));
    if (count == 0) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }
    if (sscanf(linebuffer,"%d",sectors) != 1) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }

    count = svd_read_line(fd,linebuffer,sizeof(linebuffer));
    if (count == 0) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }
    if (sscanf(linebuffer,"%d",tracks) != 1) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }

    count = svd_read_line(fd,linebuffer,sizeof(linebuffer));
    if (count == 0) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }
    if (sscanf(linebuffer,"%d",sides) != 1) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }

    count = svd_read_line(fd,linebuffer,sizeof(linebuffer));
    if (count == 0) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }
    if (sscanf(linebuffer,"%d",secsize) != 1) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }

    *secsize = (*secsize+1)*128;

    count = svd_read_line(fd,linebuffer,sizeof(linebuffer));
    if (count == 0) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }
    if (sscanf(linebuffer,"%d",wprot) != 1) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }

    *wprot = (*wprot==0)?0:0xff;

    return(E_NONE);

}

/************************************************************************
 * NAME:	svd_read_20()
 *
 * DESCR:	Read in a version 20 of an SVD file.  Picks up right
 *		after a version number was found.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
svd_read_20(int fd,struct floppy *floppy)
{
    int			sectors, tracks, sides, secsize, wprot;
    int			side, sector, track;
    int			r;
    unsigned char	blockbuf[256];
    int			indexptr;
    unsigned char	agcrbuf[256];		/* used only for AGCR format	*/

    if ( (r=svd_read_20header(fd,&sectors,&tracks,&sides,&secsize,&wprot)) != E_NONE) {
	return(r);
    }

    /* ok, now we have (sectors+1) * (tracks) of 256-byte blocks coming up	*/

    floppy->sectors = sectors;
    floppy->tracks = tracks;
    floppy->sides = sides;
    floppy->encoding = UNKNOWN_ENCODING;
    floppy->secsize = secsize;
    floppy->write_protect = wprot;

    /* allocate the track storage	*/

    floppy->side[0] = (struct track *)malloc(floppy->tracks*sizeof(struct track));

    if (floppy->sides == 2) {
	floppy->side[1] = (struct track *)malloc(floppy->tracks*sizeof(struct track));
    }

    /* allocate the sectors in each track	*/
    /* note that this allocation is sometimes too big, because a given track	*/
    /* doesn't actually have that many sectors (blank ones)...but that doesn't	*/
    /* really matter, in that the whole chunk is allocated and freed		*/
    
    for (track = 0; track < floppy->tracks; track++) {
	for (side = 0; side < floppy->sides; side++) {
	    floppy->side[side][track].sector = (struct sector *)
	                         malloc(floppy->sectors*sizeof(struct sector));
	}
    }

    /* now cruise through the blocks reading in data	*/

    for (track = 0; track < floppy->tracks; track++) {
	for (side = 0; side < floppy->sides; side++) {
	    int	blanks = 0;

	    /* first, dissect the index block for this track	*/

	    if (read(fd,blockbuf,256) != 256) {
		return(E_FATAL|E_SVD_BAD_FORMAT);
	    }

	    indexptr = 0;

	    for (sector = 0; sector < floppy->sectors; sector++) {
		int	sectortype = blockbuf[indexptr++];

		THIS_SECTOR.encoding = SVD_SECTOR_TYPE(sectortype);

		if (THIS_SECTOR.encoding == BLANK_SECTOR) {
		    /* blanks are always at the end...so if we see a blank, we're done	*/
		    /* with real data...however, we need to read in the blanks' data	*/
		    blanks++;
		}

		/* there is a "standard" amount of data per header field in the	*/
		/* track header...each format needs to advance the indexpr right.*/

		switch(THIS_SECTOR.encoding) {
		    case WD_FM:
		    case WD_MFM:
			THIS_SECTOR.id = blockbuf[indexptr++];
			THIS_SECTOR.side = blockbuf[indexptr++];
			THIS_SECTOR.sector = blockbuf[indexptr++];
			THIS_SECTOR.sizecode = blockbuf[indexptr++];
			THIS_SECTOR.headCRC = blockbuf[indexptr++];
			THIS_SECTOR.headCRC += (int)(blockbuf[indexptr++]) * 256;
			THIS_SECTOR.mark = blockbuf[indexptr++];
			THIS_SECTOR.dataCRC = blockbuf[indexptr++];
			THIS_SECTOR.dataCRC += (int)(blockbuf[indexptr++]) * 256;
			THIS_SECTOR.size = 256;
			break;
		    case H17_HSFM:
			indexptr += 19;
			if (sector == floppy->sectors - 1) {
			    indexptr += 5;		/* last sector has 5 extra		*/
			}
			THIS_SECTOR.sizecode = reflect(blockbuf[indexptr++]); /* sets the volume */
			THIS_SECTOR.id = reflect(blockbuf[indexptr++]);
			THIS_SECTOR.sector = reflect(blockbuf[indexptr++]);
			THIS_SECTOR.headCRC = reflect(blockbuf[indexptr++]);
			THIS_SECTOR.dataCRC = reflect(blockbuf[indexptr++]);
			if (THIS_SECTOR.id != 0) {
			    floppy->volume = THIS_SECTOR.sizecode;
			}
			THIS_SECTOR.size = 256;
			break;

		    case RNIB:
			/* no information is used with the RNIB format, but it is written/read	*/
			/* here just to allow the tools to work better (arguable).		*/

			THIS_SECTOR.id = blockbuf[indexptr++];
			THIS_SECTOR.side = blockbuf[indexptr++];
			THIS_SECTOR.sector = blockbuf[indexptr++];
			THIS_SECTOR.sizecode = blockbuf[indexptr++];

			if (sector != 0) {
			    indexptr += 5;
			}
			break;

	            case AGCR6x2:
	            case AGCR5x3:
		        {
			    int i;

			    for (i=0; i < 3; i++) {
				THIS_SECTOR.hdrprolog[i] = blockbuf[indexptr++];
			    }
			    THIS_SECTOR.volume = blockbuf[indexptr++];
			    THIS_SECTOR.id = blockbuf[indexptr++];
			    THIS_SECTOR.sector = blockbuf[indexptr++];
			    THIS_SECTOR.hdrchecksum = blockbuf[indexptr++];

			    /* ONLY 2 BYTES OF EPILOG - THIRD IS SVD STATIC	*/

			    for (i=0; i < 2; i++) {
				THIS_SECTOR.hdrepilog[i] = blockbuf[indexptr++];
			    }

			    for (i=0; i < 3; i++) {
				THIS_SECTOR.dataprolog[i] = blockbuf[indexptr++];
			    }
			    THIS_SECTOR.preload = blockbuf[indexptr++];
			    THIS_SECTOR.datachecksum = blockbuf[indexptr++];

			    /* standard data for AGCR	*/

			    THIS_SECTOR.size = 256;
			}
			break;

		    case RX02:
		    case BLANK_SECTOR:
		    case UNKNOWN_ENCODING:
			indexptr += 9;
			break;
		}
	    }

	    floppy->side[side][track].sectors = floppy->sectors - blanks;

	    /* now read in the data for each of the blocks in this track	*/
	    /* note that we're reading all of the data, which includes the	*/
	    /* data for the blank tracks...which is not used however...		*/

	    /* first, align to the first data block set as appropriate for the format	*/
	    /* past the first header fields, but before the data CRC/checksum...	*/
	    /* note that since we don't yet know the overall encoding, we are counting	*/
	    /* on the fact that the first sector encoding is enough to make this work	*/

	    sector = 0;

	    switch(THIS_SECTOR.encoding) {
	        case WD_FM:
	        case WD_MFM:
		    indexptr = HEADER_FIELDS_TO_CRC15;
		    break;
	        case H17_HSFM:
		    indexptr = 24;
		    break;
		case AGCR5x3:
	        case AGCR6x2:
		    indexptr = AGCR_LEADING_HEADER_BYTES;
		    break;
	        case RNIB:
		    indexptr = RNIB_LEADING_HEADER_BYTES;
		    break;
		case RX02:
	        case BLANK_SECTOR:
	        case UNKNOWN_ENCODING:
		    indexptr = HEADER_FIELDS_TO_CRC15;
		    break;
	    }

	    for (sector = 0; sector < floppy->sectors; sector++) {
		int i, j;

		if (read(fd,blockbuf,256) != 256) {
		    return(E_FATAL|E_SVD_BAD_FORMAT);
		}

		j = 0;
		for (i=indexptr; i < 256; i++) {
		    switch(THIS_SECTOR.encoding) {
		        case AGCR5x3:
		        case AGCR6x2:
			    agcrbuf[j++] = blockbuf[i];
			    break;
		        case H17_HSFM:
			    THIS_SECTOR.data[j++] = reflect(blockbuf[i]);
			    break;
		        default:
			    THIS_SECTOR.data[j++] = blockbuf[i];
			    break;
		    }
		}
		for (i = 0; i < indexptr; i++) {
		    switch(THIS_SECTOR.encoding) {
		        case AGCR5x3:
		        case AGCR6x2:
			    agcrbuf[j++] = blockbuf[i];
			    break;
		        case H17_HSFM:
			    THIS_SECTOR.data[j++] = reflect(blockbuf[i]);
			    break;
		        default:
			    THIS_SECTOR.data[j++] = blockbuf[i];
			    break;
		    }
		}

		/* AGCR format data is in a "packed bit"	*/
		/*  format that allows the hardware to generate	*/
		/*  appropriate GCR.  It needs to be unpacked	*/
		/*  before stored for the floppy sector.	*/

		switch(THIS_SECTOR.encoding) {
		    case AGCR5x3:
			a2_5x3_unpack(agcrbuf,THIS_SECTOR.data,THIS_SECTOR.preload);
			break;
		    case AGCR6x2:
			a2_6x2_unpack(agcrbuf,THIS_SECTOR.data,THIS_SECTOR.preload);
			break;
		}

		switch (THIS_SECTOR.encoding) {
		    case WD_FM:
		    case WD_MFM:
			indexptr += HEADER_FIELDS_15;		/* jump to next header fields	*/
			break;
		    case H17_HSFM:
			indexptr += 25;
			if (sector == floppy->sectors - 2) {
			    indexptr += 5;
			}
			break;
	            case AGCR6x2:
	            case AGCR5x3:
			indexptr += AGCR_SECTOR_HEADER_BYTES;
			break;
		    case RNIB:
			indexptr += RNIB_SECTOR_HEADER_BYTES;
			break;
		    case RX02:
	            case BLANK_SECTOR:
	            case UNKNOWN_ENCODING:
			indexptr += HEADER_FIELDS_15;
		    break;
		}
	    }
	}
    }

    /* set the overall encoding */

    {
	int mixed;

	floppy->encoding = floppy_overall_encoding(floppy, &mixed, -1);

	/* take care of remapping the encoded self-syncs for RNIB format	*/

	if (floppy->encoding == RNIB) {
	    rnib_selfsync_process(floppy,FALSE);
	}
    }

    return(E_NONE);
}

/************************************************************************
 * NAME:	svd_dump_20()
 *
 * DESCR:	Dumps out the floppy information in SVD v2.0 format.
 *		This is like 1.5 except for the following:
 *		- sector size in the header
 *		- write-protect in the header
 *		- trailer count included in header block
 *
 * ARGS:	fd is the file descriptor to dump on
 *		do_slide is a boolean indicating whether slide is used
 *
 * RETURNS:	0 if things OK, an error number otherwise
 *
 * NOTES:	Currently assumes single density only.
 *
 *		BIG NOTE - some things aren't currently in the file:
 *			write-protect, sides, sector/floppy density (v1.2)
 ************************************************************************/
int
svd_dump_20(struct floppy *floppy, int fd)
{
    /* this will dump out the existing floppy data in SVD format	*/

    /* for each track, dump out sectors+1 with the proper rotation	*/

    int			side;
    int			track;
    int			sector;
    int			retvalue = 0;
    int			indexptr;
    unsigned char	block[256];
    unsigned char	buffer[100];
    unsigned char	agcrbuf[256];		/* used only for AGCR format	*/

    sprintf(buffer,"%s\n","2.0");	   		   write(fd,buffer,strlen(buffer));
    sprintf(buffer,"%d\n",floppy->sectors);		   write(fd,buffer,strlen(buffer));
    sprintf(buffer,"%d\n",floppy->tracks);  		   write(fd,buffer,strlen(buffer));
    sprintf(buffer,"%d\n",floppy->sides);  		   write(fd,buffer,strlen(buffer));
    sprintf(buffer,"%d\n",floppy->secsize/128-1);          write(fd,buffer,strlen(buffer));
    sprintf(buffer,"%d\n",(floppy->write_protect==0)?0:1); write(fd,buffer,strlen(buffer));

    {
	/* take care of mapping what appears to be self-syncs in RNIB format	*/

	if (floppy->encoding == RNIB) {
	    rnib_selfsync_process(floppy,TRUE);
	}
    }
	

    for (track = 0; track < floppy->tracks; track++) {

	for (side = 0; side < floppy->sides; side++) {

	    { 
		int	i;

		for (i=0; i < 256; i++) {	/* ensures that an output file	*/
		    block[i] = 0;		/*   is always the same		*/
		}
	    }

	    indexptr = 0;

	    /* first compose the indexblock for the track	*/

	    for (sector = 0; sector < THIS_TRACK.sectors; sector++) {

		block[indexptr++] = SVD_ENCODING_FLAG(THIS_SECTOR.encoding);

		/* FOR AGCR encodings:						*/
		/* "preload" must be set at this point because it must be	*/
		/*  written with the header.  Unfortunately, the packed buffer	*/
		/*  is discarded and redone when the data for the sector is to	*/
		/*  be written.  If this runs too slowly, we'll probably want	*/
		/*  to dedicate some memory in the floppy structure so that the	*/
		/*  "pack" can be done once.					*/

		switch(THIS_SECTOR.encoding) {
		    case AGCR5x3:
			THIS_SECTOR.preload = a2_5x3_pack(THIS_SECTOR.data,agcrbuf);
			break;
		    case AGCR6x2:
			THIS_SECTOR.preload = a2_6x2_pack(THIS_SECTOR.data,agcrbuf);
			break;
		}

		switch(THIS_SECTOR.encoding) {
		    case WD_FM:
		    case WD_MFM:
			block[indexptr++] = THIS_SECTOR.id;
			block[indexptr++] = THIS_SECTOR.side;
			block[indexptr++] = THIS_SECTOR.sector;
			block[indexptr++] = THIS_SECTOR.sizecode;
			block[indexptr++] = THIS_SECTOR.headCRC & 0xff;
			block[indexptr++] = THIS_SECTOR.headCRC / 256;
			block[indexptr++] = THIS_SECTOR.mark;
			block[indexptr++] = THIS_SECTOR.dataCRC & 0xff;
			block[indexptr++] = THIS_SECTOR.dataCRC / 256;
			break;

		    case H17_HSFM:
			indexptr += 19;			/* uses pre-alignment for h17 formst	*/
			if (sector == THIS_TRACK.sectors - 1) {
			    indexptr += 5;		/* last sector has 5 extra		*/
			}
			/* first set the right volume...always zero on first track	*/
			block[indexptr++] = reflect((THIS_SECTOR.id == 0)?0:THIS_SECTOR.sizecode);
			block[indexptr++] = reflect(THIS_SECTOR.id);
			block[indexptr++] = reflect(THIS_SECTOR.sector);
			block[indexptr++] = reflect(THIS_SECTOR.headCRC) & 0xff;
			block[indexptr++] = reflect(THIS_SECTOR.dataCRC) & 0xff;
			break;

		    case RNIB:
			/* the information stuck into the RNIB header really isn't used		*/
			/* but it's thrown in there for completeness				*/
			/* it also all works together to align the header to put the trailer	*/
			/* on the last byte.							*/

			block[indexptr++] = THIS_SECTOR.id;
			block[indexptr++] = THIS_SECTOR.side;
			block[indexptr++] = THIS_SECTOR.sector;
			block[indexptr++] = THIS_SECTOR.sizecode;

			if (sector != 0) {
			    indexptr += 5;
			}
			break;

		    case AGCR6x2:
		    case AGCR5x3:
		        {
			    int i;

			    for (i=0; i < 3; i++) {
				block[indexptr++] = THIS_SECTOR.hdrprolog[i];
			    }
			    block[indexptr++] = THIS_SECTOR.volume;
			    block[indexptr++] = THIS_SECTOR.id;
			    block[indexptr++] = THIS_SECTOR.sector;
			    block[indexptr++] = THIS_SECTOR.hdrchecksum;

			    /* ONLY 2 BYTES OF EPILOG - THIRD IS SVD STATIC	*/

			    for (i=0; i < 2; i++) {
				block[indexptr++] = THIS_SECTOR.hdrepilog[i];
			    }

			    for (i=0; i < 3; i++) {
				block[indexptr++] = THIS_SECTOR.dataprolog[i];
			    }
			    block[indexptr++] = THIS_SECTOR.preload;
			    block[indexptr++] = THIS_SECTOR.datachecksum;
			}
			break;

		    case RX02:
		    case BLANK_SECTOR:
		    case UNKNOWN_ENCODING:
			indexptr += 9;		/* just blow by it	*/
			break;
		}
	    }

	    /* if the number of real sectors is lower than the "all track" sector	*/
	    /* numbers, then generate an appropriate number of blank sectors here.	*/

	    for ( ; sector < floppy->sectors; sector++) {
		block[indexptr++] = SVD_BLANK_FLAG(floppy);
		block[indexptr++] = 0x00;	/* id	-  just dummy data from here on	*/
		block[indexptr++] = 0x00;	/* side		*/
		block[indexptr++] = 0x00;	/* sector	*/
		block[indexptr++] = 0x00;	/* sizecode	*/
		block[indexptr++] = 0x00;	/* header CRC1	*/
		block[indexptr++] = 0x00;	/* header CRC2	*/
		block[indexptr++] = 0x00;	/* mark		*/
		block[indexptr++] = 0x00;	/* data CRC1	*/
		block[indexptr++] = 0x00;	/* data CRC2	*/
	    }

	    block[indexptr++] = SVD_TRAILER_FLAG(floppy);	/* type of trailer	*/

	    /* the number of bytes in the trailer follows the trailer format for those	*/
	    /* trailer formats that care...						*/

	    if (SVD_TRAILER_FLAG(floppy) != SVD_NO_TRAILER) {
		block[indexptr] = SVD_TRAILER_BYTES(floppy);
	    }

	    write(fd,block,256);

	    /* now compose and write out sectors */

	    /* first, align to the first data block set as appropriate for the format	*/
	    /* past the first header fields, but before the data CRC...			*/

	    sector = 0;

	    switch(THIS_SECTOR.encoding) {
	        case WD_FM:
	        case WD_MFM:
		    indexptr = HEADER_FIELDS_TO_CRC15;
		    break;
	        case H17_HSFM:
		    indexptr = 24;
		    break;
	        case AGCR6x2:
	        case AGCR5x3:
		    indexptr = AGCR_LEADING_HEADER_BYTES;
		    break;
	        case RNIB:
		    indexptr = RNIB_LEADING_HEADER_BYTES;
		    break;
		case RX02:
	        case BLANK_SECTOR:
	        case UNKNOWN_ENCODING:
		    indexptr = HEADER_FIELDS_TO_CRC15;
		    break;
	    }


	    for (sector = 0; sector < THIS_TRACK.sectors; sector++) {
		int	i, j;

		/* AGCR format data is in a "packed bit"	*/
		/*  format that allows the hardware to generate	*/
		/*  appropriate GCR.  It needs to be unpacked	*/
		/*  before stored for the floppy sector.	*/
		/*						*/
		/* NOTE - the "preload" is being ignored below.	*/
		/*  This is because it has already be noted and	*/
		/*  written out in the head above.		*/

		switch(THIS_SECTOR.encoding) {
		    case AGCR5x3:
			(void) a2_5x3_pack(THIS_SECTOR.data,agcrbuf);
			break;
		    case AGCR6x2:
			(void) a2_6x2_pack(THIS_SECTOR.data,agcrbuf);
			break;
		}

		j = 0;
		for (i=indexptr; i < 256; i++) {
		    switch(THIS_SECTOR.encoding) {
		        case AGCR5x3:
		        case AGCR6x2:
			    block[i] = agcrbuf[j++];
			    break;
		        case H17_HSFM:
			    block[i] = reflect(THIS_SECTOR.data[j++]);
			    break;
		        default:
			    block[i] = THIS_SECTOR.data[j++];
			    break;
		    }
		}
		for (i = 0; i < indexptr; i++) {
		    switch(THIS_SECTOR.encoding) {
		        case AGCR5x3:
		        case AGCR6x2:
			    block[i] = agcrbuf[j++];
			    break;
		        case H17_HSFM:
			    block[i] = reflect(THIS_SECTOR.data[j++]);
			    break;
		        default:
			    block[i] = THIS_SECTOR.data[j++];
			    break;
		    }
		}

		write(fd,block,256);

		switch (THIS_SECTOR.encoding) {
		    case WD_FM:
		    case WD_MFM:
			indexptr += HEADER_FIELDS_15;		/* jump to next header fields	*/
			break;
		    case H17_HSFM:
			indexptr += 25;
			if (sector == THIS_TRACK.sectors - 2) {
			    indexptr += 5;
			}
			break;
	            case AGCR6x2:
	            case AGCR5x3:
			indexptr += AGCR_SECTOR_HEADER_BYTES;
			break;
		    case RNIB:
			indexptr += RNIB_SECTOR_HEADER_BYTES;
			break;
		    case RX02:
	            case BLANK_SECTOR:
	            case UNKNOWN_ENCODING:
			indexptr += HEADER_FIELDS_15;
			break;
		}

	    }

	    /* write out some dummy sectors if there are to be blanks	*/
	    /* they'll have "blank sector data" in them...note that	*/
	    /* this data isn't really "on the disk" but it is in the	*/
	    /* floppy SVD source data file				*/

	    for ( ; sector < floppy->sectors; sector++) {
		strcpy(block,"   *** blank sector data ***   ");
		write(fd,block,256);
	    }
	}
    }

    {
	/* un-do the mapping of the data in the floppy structure	*/

	if (floppy->encoding == RNIB) {
	    rnib_selfsync_process(floppy,FALSE);
	}
    }

    return(0);
}
